<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use CodeIgniter\HTTP\CLIRequest;
use CodeIgniter\HTTP\IncomingRequest;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Psr\Log\LoggerInterface;


/**
 * LOAD ALL THE GLOBAL MODELS
 */
use App\Models\UsersModel; // Users


/**
 * LOAD ALL THE REQUIRED LIBRARIES
 */
use App\Libraries\Template; // Template Layouts

/**
 * Class BaseController
 *
 * BaseController provides a convenient place for loading components
 * and performing functions that are needed by all your controllers.
 * Extend this class in any new controllers:
 *     class Home extends BaseController
 *
 * For security be sure to declare any new methods as protected or private.
 */
class BaseController extends Controller
{
    /**
     * Instance of the main Request object.
     *
     * @var CLIRequest|IncomingRequest
     */
    protected $request;

    /**
     * An array of helpers to be loaded automatically upon
     * class instantiation. These helpers will be available
     * to all other controllers that extend BaseController.
     *
     * @var array
     */
    protected $helpers = ['url']; // url helper
    protected $template; // template settings
    protected $tokenData = []; // security token
    protected $data = []; // global data
    protected $caseInsensitiveAlphabet = 'abcdefghijklmnopqrstiuvwxyzABCDEFGHIJKLMONPQRSTUVWXYZ';

    /**
     * Constructor.
     */
    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {

        // Do Not Edit This Line
        parent::initController($request, $response, $logger);

        // Preload any models, libraries, etc, here.

        // Get a new View instance for all the availables controllers
        $this->view = \Config\Services::renderer();

        // Template Layouts render
        $this->template = new Template();

        // Load Session library
        $this->session = \Config\Services::session();

        // Load the project config data as global data
        $this->data['project']      = new \Config\Project;

        // Try to get the REQUEST locale
        $this->data['locale']       = service('request')->getLocale();

        // Set the default page script to load
        $this->data['pageScripts']  = [];


        // Check the authentication token
        if(empty($this->session->connected))
        {

            // Remove the token data from session informations
            $this->session->remove('connected');

        } // End if

        // Get the connected user profile informations
        $this->data['profile']      = (! empty($this->session->user)) ? $this->session->user : [];
        

        // Set the menu global data
        $this->data['activeMenu']   = [
            'home'      => [
                'dashboard'     => '',
                'reports'       => '',
                'profile'       => '',
            ],
            'purchases' => [
                'suppliers'     => '',
                'requests'      => '',
                'orders'        => '',
                'invoices'      => '',
                'deliveries'    => '',
                'logs'          => '',
            ],
            'sales'     => [
                'customers'     => '',
                'quotes'        => '',
                'deliveries'    => '',
                'orders'        => '',
                'invoices'      => '',
            ],
            'products'  => [
                'tools'         => '',
                'consumables'   => '',
            ],
            'stocks'    => [
                'tools'         => '',
                'consumables'   => '',
                'assets'        => '',
            ],
            'projects'  => [
                'projects'      => '',
                'professions'   => '',
                'managers'      => '',
                'supervisors'   => '',
            ],
            'settings'  => [
                'currencies'    => '',
                'roles'         => '',
                'permissions'   => '',
                'users'         => '',
            ],
            'audits'    => [
                'audits'        => '',
            ],
        ];


        /* Localization */
        $generalLanguagePath = "Language/{$this->data['locale']}/General.php";
        $generalLanguageFile = (file_exists(APPPATH . $generalLanguagePath)) ? require_once APPPATH . $generalLanguagePath : '';
        $projectLanguagePath = "Language/{$this->data['locale']}/Project.php";
        $projectLanguageFile = (file_exists(APPPATH . $projectLanguagePath)) ? require_once APPPATH . $projectLanguagePath : '';
        $datesLanguagePath = "Language/{$this->data['locale']}/Dates.php";
        $datesLanguageFile = (file_exists(APPPATH . $datesLanguagePath)) ? require_once APPPATH . $datesLanguagePath : '';

        // Make localization files available through views files
        $this->data['langFile'] = [
            'general'   => $generalLanguageFile,
            'project'   => $projectLanguageFile,
            'dates'     => $datesLanguageFile,
        ];
    }


    /**
     * Return a boolean that corresponds
     * To the question 'does user have this permission'
     * According to the user role on Auth
     *
     * @param string $permission
     *
     * @return bool $response
     */
    public function _hasPermission(string $permission = ''):?bool
    {

        // Set the default answer to return
        $response = false;

        // Get the permission
        $permission = (! empty($permission)) ? mb_strtoupper($permission) : '';
        $rootPermission = mb_strtoupper($this->data['project']->root_permission);
        $userPermissions = (! empty($this->session->user['permissions'])) ? $this->session->user['permissions'] : [];

        // Process only if the permission is not empty
        if(! empty($permission))
        {

            // Check if the user has the specified permission
            if(in_array($permission, $userPermissions) || in_array($rootPermission, $userPermissions)) {

                // Set the answer
                $response = true;

            } // End if

        } // End if

        // Return the answer
        return $response;

    } // end::_hasPermission


    /**
     * Return a six digits password
     * That contains 4 numbers and 2 letters
     *
     * @return string $code
     */
    public function _generatePassword()
    {

        // Get a random number of four digits
        $randomNumber = mt_rand(1001, 9999);

        // Get a random letter from the alphabet
        $randomLetter1 = substr($this->caseInsensitiveAlphabet, mt_rand(0, 52), 1);

        // Get another random letter from the alphabet
        $randomLetter2 = substr($this->caseInsensitiveAlphabet, mt_rand(0, 52), 1);

        // Set a full code with all the random value get
        $code = $randomLetter1 . '' . $randomNumber . '' . $randomLetter2;

        // Shuffle the code
        return str_shuffle($code);

    } // end::_generatePassword
}
